import * as fflate from 'https://cdn.jsdelivr.net/npm/fflate@0.8.2/esm/browser.js';

/**
 * Crée l'objet de métadonnées h5p.json.
 * @param {string} title - Le titre de la vidéo.
 * @returns {object} L'objet h5p.json.
 */
function createH5PJson(title) {
    return {
        "title": title,
        "language": "fr",
        "mainLibrary": "H5P.InteractiveVideo",
        "embedTypes": ["iframe"],
        "license": "U",
        "preloadedDependencies": [
            { "machineName": "H5P.InteractiveVideo", "majorVersion": 1, "minorVersion": 26 },
            { "machineName": "H5P.SingleChoiceSet", "majorVersion": 1, "minorVersion": 11 }
        ]
    };
}

/**
 * Fonction principale qui orchestre la création du package .h5p.
 * @param {object} config - L'objet de configuration complet chargé en mémoire.
 */
export function generateH5PPackage(config) {
    if (!config || !config.editorContent) {
        alert("Veuillez d'abord charger ou créer une configuration avec du contenu éditable.");
        return;
    }

    try {
        // --- NOUVELLE LOGIQUE DE TRANSFORMATION ---
        // Nous allons recréer l'objet contentJson à partir de la configuration en mémoire
        // pour garantir qu'il est parfaitement structuré, surtout pour "adaptivity".
        
        const sourceContent = config.editorContent;
        const transformedInteractions = sourceContent.interactiveVideo.assets.interactions.map(interaction => {
            const newInteraction = { ...interaction };

            // Récupère la valeur de retour en arrière depuis l'objet adaptivity existant
            const seekBackTime = newInteraction.adaptivity?.wrong?.seek;

            // Construit le nouvel objet adaptivity
            const newAdaptivity = {
                correct: {}, // Toujours un objet vide pour l'instant
                wrong: {
                    allowOptOut: false,
                    message: ""
                },
                requireCompletion: newInteraction.adaptivity?.requireCompletion ?? false
            };

            // CONDITION : Ajoute la propriété "seekTo" UNIQUEMENT si la valeur est supérieure à 0
            if (seekBackTime && seekBackTime > 0) {
                newAdaptivity.wrong.seekTo = seekBackTime;
            }
            
            // Remplace l'ancien objet adaptivity par le nouveau, bien formé
            newInteraction.adaptivity = newAdaptivity;

            // Supprime la propriété "seek" de l'ancien format si elle existe encore
            // (nettoyage pour éviter les confusions)
            if (newInteraction.adaptivity.wrong.seek) {
                delete newInteraction.adaptivity.wrong.seek;
            }

            return newInteraction;
        });

        // Construit l'objet content.json final avec les interactions transformées
        const contentJson = {
            ...sourceContent,
            interactiveVideo: {
                ...sourceContent.interactiveVideo,
                assets: {
                    ...sourceContent.interactiveVideo.assets,
                    interactions: transformedInteractions
                }
            }
        };

        // --- Le reste de la fonction est inchangé ---
        const videoTitle = contentJson.interactiveVideo.video.startScreenOptions.title || "video-interactive";

        const fileName = videoTitle
            .replace(/\s+/g, '-')
            .replace(/[\\/:*?"<>|]/g, '')
            .toLowerCase();

        const h5pJson = createH5PJson(videoTitle);

        const filesToZip = {
            'h5p.json': [
                fflate.strToU8(JSON.stringify(h5pJson, null, 2)),
                { level: 0 }
            ],
            'content/content.json': [
                fflate.strToU8(JSON.stringify(contentJson, null, 2)),
                { level: 6 }
            ]
        };

        fflate.zip(filesToZip, (err, data) => {
            if (err) {
                throw err;
            }
            const blob = new Blob([data], { type: 'application/zip' });
            const url = URL.createObjectURL(blob);
            const link = document.createElement('a');
            link.href = url;
            link.download = `${fileName}.h5p`;
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
            URL.revokeObjectURL(url);
            console.log(`Package H5P '${fileName}.h5p' généré et téléchargé.`);
        });

    } catch (error) {
        console.error("Erreur lors de la génération du package H5P :", error);
        alert("Une erreur est survenue. Vérifiez que la configuration est valide.");
    }
}